﻿-- =============================================
-- This script upgrades the Death Indicator
-- =============================================

SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
SET ANSI_PADDING ON
GO
IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[hips].[DeathIndicator]') AND type in (N'U'))
BEGIN
CREATE TABLE [hips].[DeathIndicator](
	[DeathIndicatorId] [int] NOT NULL,
	[Description] [varchar](100) NOT NULL,
	[DateCreated] [datetime] NOT NULL,
	[UserCreated] [varchar](256) NOT NULL,
	[DateModified] [datetime] NOT NULL,
	[UserModified] [varchar](256) NOT NULL,
 CONSTRAINT [PK_DeathIndicator] PRIMARY KEY CLUSTERED 
(
	[DeathIndicatorId] ASC
)WITH (PAD_INDEX  = OFF, STATISTICS_NORECOMPUTE  = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS  = ON, ALLOW_PAGE_LOCKS  = ON) ON [PRIMARY]
) ON [PRIMARY]
END
GO
SET ANSI_PADDING OFF
GO

IF NOT EXISTS (SELECT DeathIndicatorId FROM hips.DeathIndicator WHERE DeathIndicatorId = -1)
BEGIN
	INSERT hips.DeathIndicator (DeathIndicatorId, Description, DateCreated, UserCreated, DateModified, UserModified)
	VALUES (-1, 'Unknown', GETDATE(), 'HIPS', GETDATE(), 'HIPS')
END
IF NOT EXISTS (SELECT DeathIndicatorId FROM hips.DeathIndicator WHERE DeathIndicatorId = 1)
BEGIN
	INSERT hips.DeathIndicator (DeathIndicatorId, Description, DateCreated, UserCreated, DateModified, UserModified)
	VALUES (1, 'ValidDate', GETDATE(), 'HIPS', GETDATE(), 'HIPS')
END
IF NOT EXISTS (SELECT DeathIndicatorId FROM hips.DeathIndicator WHERE DeathIndicatorId = 2)
BEGIN
	INSERT hips.DeathIndicator (DeathIndicatorId, Description, DateCreated, UserCreated, DateModified, UserModified)
	VALUES (2, 'InvalidDate', GETDATE(), 'HIPS', GETDATE(), 'HIPS')
END
GO

IF NOT EXISTS (SELECT * from sys.columns WHERE name = 'DeathIndicatorId' and object_id = OBJECT_ID(N'[hips].[PatientMaster]'))
BEGIN
	ALTER TABLE [hips].[PatientMaster] ADD [DeathIndicatorId] [int] NULL
END


IF NOT EXISTS (SELECT * FROM ::fn_listextendedproperty(N'MS_Description' , N'SCHEMA',N'hips', N'TABLE',N'PatientMaster', N'COLUMN',N'DeathIndicatorId'))
EXEC sys.sp_addextendedproperty @name=N'MS_Description', @value=N'Death indicator as linked to Date of Death' , @level0type=N'SCHEMA',@level0name=N'hips', @level1type=N'TABLE',@level1name=N'PatientMaster', @level2type=N'COLUMN',@level2name=N'DeathIndicatorId'
GO



IF NOT EXISTS (SELECT * FROM sys.foreign_keys WHERE object_id = OBJECT_ID(N'[hips].[FK_PatientMaster_DeathIndicator]') AND parent_object_id = OBJECT_ID(N'[hips].[PatientMaster]'))
ALTER TABLE [hips].[PatientMaster]  WITH CHECK ADD  CONSTRAINT [FK_PatientMaster_DeathIndicator] FOREIGN KEY([DeathIndicatorId])
REFERENCES [hips].[DeathIndicator] ([DeathIndicatorId])
GO
IF  EXISTS (SELECT * FROM sys.foreign_keys WHERE object_id = OBJECT_ID(N'[hips].[FK_PatientMaster_DeathIndicator]') AND parent_object_id = OBJECT_ID(N'[hips].[PatientMaster]'))
ALTER TABLE [hips].[PatientMaster] CHECK CONSTRAINT [FK_PatientMaster_DeathIndicator]
GO

DROP PROCEDURE [hips].[PatientMasterGet]
GO
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
-- =============================================
-- Author:        Andrew Jacks
-- Create date: 12 October 2012
-- Description:   Gets a specific record from the Patient Master table.
--
-- Lookup Types Supported:
--    1. PatientMasterId (e.g. 1234567)
--    2. StatePatientId (e.g. '100000090751')
--    3. Ihi (e.g. '8003604567901177')
--    4. HospitalId and Mrn  (e.g. 71, '01234567')
--    5. HospitalCode and CodeSystemId and Mrn ('FMC', 2, '01234567')
--    6. Demographic search using all of:
--         CurrentLastName, CurrentFirstNames, DateOfBirth, CurrentSexId
--       and one of:
--           DvaNumber, MedicareNumber
--       and optionally MedicareIrn.
--       For demographic search the business logic must pass a valid medicare
--       number or null, but avoid passing a special value like '8' or '0',
--       which could match records that it should not match.
--
-- Modified: 23/11/2012 sbiber added IHI and StatePatientId lookup
-- Modified: 18/12/2012 sbiber removed PcehrAdvertised
-- Modified: 24/01/2012 sbiber added conditions to prevent returning all patients
-- Modified: 07/05/2013 sbiber tightened use of demographic search
-- Modified: 16/06/2013 pstall01 added date of death indicator
-- =============================================
CREATE PROCEDURE [hips].[PatientMasterGet] 
(
	@PatientMasterId INT = NULL,
	@HospitalId INT = NULL,
	@HospitalCode VARCHAR(64) = NULL,
	@CodeSystemId INT = NULL,
	@Mrn VARCHAR(20) = NULL,
	@CurrentLastName VARCHAR(80) = NULL,
	@CurrentFirstNames VARCHAR(80) = NULL,
	@DateOfBirth DATETIME = NULL,
	@CurrentSexId INT = NULL,
	@MedicareNumber VARCHAR(12) = NULL,
	@MedicareIrn VARCHAR(3) = NULL,
	@DvaNumber VARCHAR(18) = NULL,
	@Ihi VARCHAR(16) = NULL,
	@StatePatientId VARCHAR(64) = NULL
)
AS
BEGIN
	IF @HospitalId IS NULL AND @CodeSystemId IS NOT NULL AND @HospitalCode IS NOT NULL 
	BEGIN
	  SELECT @HospitalId = [HospitalId]
	  FROM [hips].[HospitalCode]
	  WHERE [CodeSystemId] = @CodeSystemId
	  AND [Code] = @HospitalCode
	END


	IF @PatientMasterID IS NULL
	BEGIN
		IF @HospitalId IS NOT NULL AND @Mrn IS NOT NULL
		BEGIN
			SELECT	@PatientMasterId = [PatientMasterId]
			FROM	[hips].[HospitalPatient]
			WHERE	[HospitalId] = @HospitalId
			AND		[Mrn] = @Mrn
		END
		ELSE IF @Ihi IS NOT NULL
		BEGIN
			SELECT	@PatientMasterId = [PatientMasterId]
			FROM	[hips].[PatientMasterIhi]
			WHERE	[Ihi] = @Ihi
		END
		ELSE IF @StatePatientId IS NOT NULL
		BEGIN
			SELECT	@PatientMasterId = [PatientMasterId]
			FROM	[hips].[PatientMaster]
			WHERE	[StatePatientId] = @StatePatientId
		END				
	END	

	IF  @PatientMasterId IS NULL
	AND @Mrn IS NULL
	AND @StatePatientId IS NULL
	AND @Ihi IS NULL
	AND @CurrentFirstNames IS NOT NULL 
	AND @CurrentLastName IS NOT NULL
	AND @DateOfBirth IS NOT NULL
	AND (  @MedicareNumber IS NOT NULL
		OR @DvaNumber IS NOT NULL
		)
	BEGIN
		SELECT pm.[PatientMasterId]
			 , COALESCE (pmi.RegisteredSexId,pm.CurrentSexId, -1) AS RegisteredSexId
			 , COALESCE (pm.CurrentSexId, -1) AS CurrentSexId
			 , pmi.Ihi
			 , COALESCE (pmi.IhiStatusId, -1) AS IhiStatusId
			 , ihis.Description AS IhiStatus
			 , COALESCE (pmi.IhiRecordStatusId, -1) AS IhiRecordStatusId
			 , pmi.[RegisteredFamilyName]
			 , pmi.[RegisteredGivenName]
			 , irs.Description AS IhiRecordStatus
			 , pmi.DateLastValidated AS IhiLastValidated
			 , pm.[MedicareNumber]
			 , pm.[MedicareIrn]
			 , pm.[IsMedicareNumberValid]
			 , pm.[DvaNumber]
			 , pm.[StatePatientId]
			 , pm.[DateOfBirth]
			 , pm.[DateOfDeath]
			 , pm.[DeathIndicatorId]
			 , pm.[DateCreated]
			 , pm.[UserCreated]
			 , pm.[DateModified]
			 , pm.[UserModified]
        FROM	[hips].[PatientMaster] AS pm
        LEFT JOIN [hips].[PatientMasterIhi] AS pmi ON pmi.[PatientMasterID] = pm.[PatientMasterID]
        LEFT JOIN [hips].HospitalPatient AS p ON pm.PatientMasterID = p.PatientMasterID
		LEFT JOIN [hips].[Hospital] AS h ON h.HospitalID = p.HospitalID
		LEFT JOIN [hips].PatientMasterName pmn ON pmn.PatientMasterId = pm.PatientMasterId AND pmn.NameTypeId = 2
		LEFT JOIN [hips].IhiRecordStatus AS irs on irs.IhiRecordStatusId = pmi.IhiRecordStatusId
		LEFT JOIN [hips].IhiStatus AS ihis on ihis.IhiStatusId = pmi.IhiStatusId
		
        WHERE	(pmn.[FamilyName] = @CurrentLastName)
		AND		(pmn.[GivenNames] = @CurrentFirstNames)
		AND		(pm.[DateOfBirth] = @DateOfBirth)
		AND		(pm.[CurrentSexID] = @CurrentSexID)
		AND		(@DvaNumber IS NULL OR pm.[DvaNumber] = @DvaNumber)
		-- The business logic must ensure that special values (invalid/ineligible) are passed in as null to avoid matching
		AND		(@MedicareNumber IS NULL OR pm.[MedicareNumber] = @MedicareNumber)

		-- Do match even if the record has a null sequence number
		AND		(@MedicareIrn IS NULL OR pm.[MedicareIrn] IS NULL OR pm.[MedicareIrn] = @MedicareIrn)
	END
	ELSE  -- Lookup by @PatientMasterID
	BEGIN
		SELECT pm.[PatientMasterID]
			 , COALESCE (pmi.RegisteredSexId,pm.CurrentSexId, -1) AS RegisteredSexId
			 , COALESCE (pm.CurrentSexId, -1) AS CurrentSexId
			 , pmi.Ihi
			 , COALESCE (pmi.IhiStatusId, -1) AS IhiStatusId
			 , ihis.Description AS IhiStatus
			 , COALESCE (pmi.IhiRecordStatusId, -1) AS IhiRecordStatusId
			 , irs.Description AS IhiRecordStatus
			 , pmi.[RegisteredFamilyName]
			 , pmi.[RegisteredGivenName]
			 , pmi.DateLastValidated AS IhiLastValidated
			 , pm.[MedicareNumber]
			 , pm.[MedicareIrn]
			 , pm.[IsMedicareNumberValid]
			 , pm.[DvaNumber]
			 , pm.[StatePatientId]
			 , pm.[DateOfBirth]
			 , pm.[DateOfDeath]
			 , pm.[DeathIndicatorId]
			 , pm.[DateCreated]
			 , pm.[UserCreated]
			 , pm.[DateModified]
			 , pm.[UserModified]         
	     FROM	[hips].[PatientMaster] AS pm
         LEFT JOIN [hips].[PatientMasterIhi] AS pmi ON pmi.[PatientMasterID] = pm.[PatientMasterID]
         LEFT JOIN [hips].IhiRecordStatus AS irs on irs.IhiRecordStatusId = pmi.IhiRecordStatusId
		 LEFT JOIN [hips].IhiStatus AS ihis on ihis.IhiStatusId = pmi.IhiStatusId

        WHERE	pm.[PatientMasterID] = @PatientMasterID
		AND     (@DateOfBirth IS NULL OR @DateOfBirth = [DateOfBirth])
	END
END
GO


DROP PROCEDURE [hips].[PatientMasterInsert]
GO
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
-- =============================================
-- Author:        Gordon Massey
-- Create date: 11 Oct 2012
-- Description:   Inserts a new record into the Patient Master Table
-- Modified: 23/11/2012 sbiber added StatePatientId
-- Modified: 18/12/2012 sbiber removed PcehrAdvertised
-- Modified: 16/06/2013 pstall01 added date of death indicator
-- =============================================
CREATE PROCEDURE [hips].[PatientMasterInsert] 
(

	@CurrentSexId INT,
	@DeathIndicatorId INT = null,
	@DateOfBirth DATETIME,
	@DateOfDeath DATETIME = null,
	@MedicareNumber VARCHAR(12) = null,
	@MedicareIrn VARCHAR(3) = null,
	@IsMedicareNumberValid INT = null,
	@DvaNumber VARCHAR(18) = NULL,
	@StatePatientId VARCHAR(64) = NULL,
	@UserModified varchar(256)
)
AS
BEGIN
    SET NOCOUNT ON
	
    DECLARE @intErrorCode INT
		,@PatientMasterId INT

    SET @intErrorCode = 0

    INSERT
     INTO [hips].[PatientMaster]
         ( 
           [CurrentSexId]
		 , [DeathIndicatorId]
         , [DateOfBirth]
         , [DateOfDeath]
         , [MedicareNumber]
         , [MedicareIrn]
         , [IsMedicareNumberValid]
         , [DvaNumber]
		 , [StatePatientId]
         , [DateCreated]
         , [UserCreated]
         , [DateModified]
         , [UserModified]
         )
    VALUES
         ( 
           @CurrentSexId
		 , @DeathIndicatorId 
         , @DateOfBirth
         , @DateOfDeath
         , @MedicareNumber
         , @MedicareIrn
         , @IsMedicareNumberValid
         , @DvaNumber
		 , @StatePatientId
         , GETDATE()
         , @UserModified
         , GETDATE()
         , @UserModified
         )

    SELECT @intErrorCode = @@ERROR 
		, @PatientMasterId = SCOPE_IDENTITY()
    IF @intErrorCode = 0
    BEGIN
		SELECT pm.[PatientMasterID]
			 , COALESCE (pmi.RegisteredSexId,pm.CurrentSexId, -1) AS RegisteredSexId
			 , COALESCE (pm.CurrentSexId, -1) AS CurrentSexId
			 , pmi.Ihi
			 , COALESCE (pmi.IhiStatusId, -1) AS IhiStatusId
			 , ihis.Description AS IhiStatus
			 , COALESCE (pmi.IhiRecordStatusId, -1) AS IhiRecordStatusId
			 , pmi.[RegisteredFamilyName]
			 , pmi.[RegisteredGivenName]
		     , irs.Description AS IhiRecordStatus
			 , pmi.DateLastValidated AS IhiLastValidated
			 , pm.[MedicareNumber]
			 , pm.[MedicareIrn]
			 , pm.[IsMedicareNumberValid]
			 , pm.[DvaNumber]
			 , pm.[StatePatientId]
			 , pm.[DateOfBirth]
			 , pm.[DateOfDeath]
			 , pm.[DeathIndicatorId]
			 , pm.[DateCreated]
			 , pm.[UserCreated]
			 , pm.[DateModified]
			 , pm.[UserModified]         
	     FROM	[hips].[PatientMaster] AS pm
         LEFT JOIN [hips].[PatientMasterIhi] AS pmi ON pmi.[PatientMasterID] = pm.[PatientMasterID]
         LEFT JOIN [hips].IhiRecordStatus AS irs on irs.IhiRecordStatusId = pmi.IhiRecordStatusId
		 LEFT JOIN [hips].IhiStatus AS ihis on ihis.IhiStatusId = pmi.IhiStatusId

        WHERE	pm.[PatientMasterID] = @PatientMasterID
    END

    RETURN(@intErrorCode)
END
GO


DROP PROCEDURE [hips].[PatientMasterUpdate]
GO
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
-- =============================================
-- Author:        Andrew
-- Create date: 19 October 2012
-- Description:   Updates an existing record in the Patient Master  Table
-- Modified: 18/12/2012 sbiber removed PcehrAdvertised
-- Modified: 16/06/2013 pstall01 added date of death indicator
-- =============================================
CREATE PROCEDURE [hips].[PatientMasterUpdate] 
(
    @PatientMasterId INT,
	@CurrentSexId INT,
	@DeathIndicatorId INT = null,
	@DateOfBirth DATETIME,
	@DateOfDeath DATETIME = null,
	@MedicareNumber VARCHAR(12) = null,
	@MedicareIrn VARCHAR(3) = null,
	@IsMedicareNumberValid INT = null,
	@DvaNumber VARCHAR(18) = NULL,	 
	@StatePatientId VARCHAR(64) = NULL, 
	@UserModified VARCHAR(256),
	@DateModified datetime
)
AS
BEGIN
    SET NOCOUNT ON
 DECLARE @intErrorCode int

    SET @intErrorCode = 0

    IF NOT EXISTS (SELECT TOP 1 [DateModified]
            	FROM [hips].[PatientMaster]
				WHERE [PatientMasterId] = @PatientMasterId AND [DateModified] = @DateModified)
    BEGIN
        RAISERROR 50001 'Record has already been updated!'
        SET @intErrorCode = @@ERROR
    END
 IF @intErrorCode = 0
    BEGIN
        UPDATE [hips].[PatientMaster]
           SET [CurrentSexId] = @CurrentSexId
			  ,[DeathIndicatorId] = @DeathIndicatorId 
			  ,[DateOfBirth] = @DateOfBirth
			  ,[DateOfDeath] = @DateOfDeath
			  ,[MedicareNumber] = @MedicareNumber
			  ,[MedicareIrn] = @MedicareIrn
			  ,[IsMedicareNumberValid] = @IsMedicareNumberValid
			  ,[DvaNumber] = @DvaNumber
			  ,[StatePatientId] = @StatePatientId
             , [DateModified] = GetDate()
             , [UserModified] = @UserModified
         WHERE [PatientMasterID] = @PatientMasterID
        SELECT @intErrorCode = @@ERROR
    END

    IF @intErrorCode = 0
    BEGIN
		SELECT pm.[PatientMasterID]
			 , COALESCE (pmi.RegisteredSexId,pm.CurrentSexId, -1) AS RegisteredSexId
			 , COALESCE (pm.CurrentSexId, -1) AS CurrentSexId
			 , pmi.Ihi
			 , COALESCE (pmi.IhiStatusId, -1) AS IhiStatusId
			 , ihis.Description AS IhiStatus
			 , COALESCE (pmi.IhiRecordStatusId, -1) AS IhiRecordStatusId
			 , pmi.[RegisteredFamilyName]
			 , pmi.[RegisteredGivenName]
			 , irs.Description AS IhiRecordStatus
			 , pmi.DateLastValidated AS IhiLastValidated
			 , pm.[MedicareNumber]
			 , pm.[MedicareIrn]
			 , pm.[IsMedicareNumberValid]
			 , pm.[DvaNumber]
			 , pm.[StatePatientId]
			 , pm.[DateOfBirth]
			 , pm.[DateOfDeath]
			 , pm.[DeathIndicatorId]
			 , pm.[DateCreated]
			 , pm.[UserCreated]
			 , pm.[DateModified]
			 , pm.[UserModified]         
	     FROM	[hips].[PatientMaster] AS pm
         LEFT JOIN [hips].[PatientMasterIhi] AS pmi ON pmi.[PatientMasterID] = pm.[PatientMasterID]
         LEFT JOIN [hips].IhiRecordStatus AS irs on irs.IhiRecordStatusId = pmi.IhiRecordStatusId
		 LEFT JOIN [hips].IhiStatus AS ihis on ihis.IhiStatusId = pmi.IhiStatusId

        WHERE	pm.[PatientMasterID] = @PatientMasterID
    END

    RETURN(@intErrorCode)

END
GO


-- Grant Permissions on the Changed Procedures
GRANT EXECUTE ON [hips].[PatientMasterGet] TO [RunStoredProcedure]
GRANT EXECUTE ON [hips].[PatientMasterInsert] TO [RunStoredProcedure]
GRANT EXECUTE ON [hips].[PatientMasterUpdate] TO [RunStoredProcedure]
GO
